<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Rj;
use App\Models\Social;
use App\Rules\FileTypeValidate;
use Illuminate\Http\Request;

class ManageRjController extends Controller
{
    public function index()
    {
        $pageTitle = 'All Radio Jockeys';
        $rjs       = Rj::searchable(['name', 'email', 'contact_no'])->latest()->paginate(getPaginate());
        return view('admin.rj.index', compact('pageTitle', 'rjs'));
    }

    public function new()
    {
        $pageTitle = 'New Radio Jockey';
        return view('admin.rj.new', compact('pageTitle'));
    }

    public function edit($id)
    {
        $pageTitle         = 'Update Radio Jockey';
        $rj                = Rj::findOrFail($id);
        $rjEducationCount  = $rj->education ? count($rj->education) : 1;
        $rjExperienceCount = $rj->experience ? count($rj->experience) : 1;

        return view('admin.rj.edit', compact('pageTitle', 'rj', 'rjEducationCount', 'rjExperienceCount'));
    }

    public function store(Request $request, $id = 0)
    {


        $this->validation($request, $id);

        if ($id) {
            $rj           = Rj::findOrFail($id);
            $notification = 'Rj updated successfully';
        } else {
            $rj           = new Rj();
            $notification = 'Rj added successfully';
        }

        if ($request->hasFile('image')) {
            $fileName  = fileUploader($request->image, getFilePath('rj'), getFileSize('rj'), @$rj->image);
            $rj->image = $fileName;
        }

        $gallery = $id ? $rj->gallery : [];

        if ($request->hasFile('gallery')) {
            foreach ($request->gallery as $singleImage) {
                $gallery[] = fileUploader($singleImage, getFilePath('gallery'), getFileSize('gallery'));
            }
        }

        $rj->name        = $request->name;
        $rj->designation = $request->designation;
        $rj->email       = $request->email;
        $rj->contact_no  = $request->contact_no;
        $rj->about       = $request->about;
        $rj->education   = $request->education;
        $rj->experience  = $request->experience;
        $rj->about       = $request->about;
        $rj->gallery     = $gallery;
        $rj->save();

        $notify[] = ['success', $notification];
        return back()->withNotify($notify);
    }

    public function socialIndex($id)
    {
        $pageTitle = 'Social Icons';
        $rj        = Rj::findOrFail($id);
        $socials   = Social::where('rj_id', $rj->id)->latest()->get();
        return view('admin.rj.social', compact('pageTitle', 'rj', 'socials'));
    }

    public function socialStore(Request $request, $rjId, $id = 0)
    {
        $request->validate([
            'title' => 'required|max:255',
            'icon'  => 'required|max:255',
            'url'   => 'required|url|max:255',
        ]);

        $rj = Rj::findOrFail($rjId);

        if ($id) {
            $social       = Social::where('id', $id)->where('rj_id', $rj->id)->findOrFail($id);
            $notification = 'Social icon updated successfully';
        } else {
            $social        = new Social();
            $social->rj_id = $rj->id;
            $notification  = 'Social icon added successfully';
        }

        $social->title = $request->title;
        $social->url   = $request->url;
        $social->icon  = $request->icon;
        $social->save();

        $notify[] = ['success', $notification];

        return to_route('admin.rj.social.index', $rj->id)->withNotify($notify);
    }

    public function removeGallery($id, $imageName)
    {
        $rj         = Rj::findOrFail($id);
        $gallery    = [];
        $imageCheck = in_array($imageName, $rj->gallery);

        if (!$imageCheck) {
            $notify[] = ['error', 'Image not found'];
            return back()->withNotify($notify);
        }

        foreach ($rj->gallery as $singleImage) {
            if ($singleImage != $imageName) {
                $gallery[] = $singleImage;
            }
        }

        $rj->gallery = $gallery;
        $rj->save();

        fileManager()->removeFile(getFilePath('gallery') . '/' . $imageName);

        $notify[] = ['success', 'Image removed successfully'];
        return back()->withNotify($notify);
    }

    protected function validation($request, $id)
    {
        $imageValidation = $id ? 'nullable' : 'required';

        $request->validate([
            'name'                        => 'required|max:255',
            'designation'                 => 'required|max:255',
            'email'                       => 'required|max:255|unique:rjs,email,' . $id,
            'contact_no'                  => 'required|max:255|unique:rjs,contact_no,' . $id,
            'about'                       => 'required',

            'education'                   => 'required|array|min:1',
            'education.*.name'            => 'required|max:255',
            'education.*.about'           => 'required',
            'education.*.form_year'       => 'required|date_format:Y|before:education.*.to_year',
            'education.*.to_year'         => 'required|date_format:Y|after:education.*.form_year',

            'experience'                  => 'required|array|min:1',
            'experience.*.name'           => 'required|max:255',
            'experience.*.responsibility' => 'required',
            'experience.*.form_year'      => 'required|date_format:Y|before:experience.*.to_year',
            'experience.*.to_year'        => 'required|date_format:Y|after:experience.*.form_year',

            'experience'                  => 'required|array|min:1',
            'image'                       => [$imageValidation, 'image', 'max:2048', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
            'gallery.*'                   => ['nullable', 'image', 'max:2048', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
        ]);
    }
}
